<?php
require_once 'config.php';        // Nạp file cấu hình
require_once 'db_connection.php'; // Nạp file kết nối DB và hàm tiện ích

// Hàm để làm sạch memberID theo chuẩn của AppGini (chỉ cho phép chữ/số/dấu gạch dưới/dấu chấm/khoảng trắng/dấu gạch ngang)
function safeMemberID($memberID) {
    return strtolower(preg_replace('/[^\w\. @\-]/', '', trim($memberID)));
}

// HÀM MỚI: Mô phỏng logic password_match của AppGini
// Bạn cần kiểm tra hàm password_match() thực tế trong lib.php của AppGini
// để đảm bảo logic này khớp chính xác.
function verifyAppGiniPassword($rawPassword, $storedHash) {
    // Nếu hash được lưu trữ là MD5 (32 ký tự hex)
    if (preg_match('/^[a-f0-9]{32}$/i', $storedHash)) {
        return (md5($rawPassword) === $storedHash);
    }

    // Nếu hash được lưu trữ là một định dạng băm mạnh hơn (ví dụ: bcrypt, argon2id)
    // password_verify() của PHP sẽ tự động xử lý salt và thuật toán
    return password_verify($rawPassword, $storedHash);
}


// Hàm để đăng nhập người dùng và tạo JWT
function loginUser($username, $password) {
    $conn = getDbConnection(); // Lấy kết nối cơ sở dữ liệu

    $safeUsername = safeMemberID($username);

    // Bước 1: Lấy hash mật khẩu, trạng thái phê duyệt và bị cấm từ DB
    // Vẫn dùng `memberID` làm cột username và `passMD5` là cột mật khẩu
    $stmt = $conn->prepare("SELECT `memberID`, `groupID`, `passMD5`, `isApproved`, `isBanned` FROM `" . MEMBERSHIP_USERS_TABLE . "` WHERE `memberID` = ?");
    if (!$stmt) {
        error_log("Error preparing login query: " . $conn->error);
        $conn->close();
        return false;
    }
    $stmt->bind_param("s", $safeUsername);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
    $conn->close();

    if (!$user) {
        // Không tìm thấy người dùng
        error_log("Login failed: User '{$safeUsername}' not found.");
        return false;
    }

    // Bước 2: Kiểm tra trạng thái isApproved và isBanned
    if ($user['isApproved'] != 1 || $user['isBanned'] != 0) {
        error_log("Login failed: User '{$safeUsername}' is not approved or is banned. isApproved: {$user['isApproved']}, isBanned: {$user['isBanned']}");
        return false;
    }

    // Bước 3: So sánh mật khẩu bằng hàm mô phỏng password_match
    if (!verifyAppGiniPassword($password, $user['passMD5'])) {
        error_log("Login failed: Incorrect password for user '{$safeUsername}'.");
        return false;
    }

    // Nếu đến đây, mật khẩu và trạng thái người dùng đều hợp lệ
    // Tạo JWT token
    $header = base64_encode(json_encode(['alg' => 'HS256', 'typ' => 'JWT']));
    $payload = base64_encode(json_encode([
        'userID' => $user['memberID'],
        'username' => $username,
        'groupID' => $user['groupID'],
        'exp' => time() + JWT_EXPIRATION_SECONDS
    ]));
    $signature = base64_encode(hash_hmac('sha256', "$header.$payload", JWT_SECRET_KEY, true));
    $token = "$header.$payload.$signature";
    return $token;
}

// Hàm để xác thực người dùng dựa trên JWT token từ header Authorization
function authenticateUser() {
    $authHeader = isset($_SERVER['HTTP_AUTHORIZATION']) ? $_SERVER['HTTP_AUTHORIZATION'] : '';

    if (empty($authHeader) || !preg_match('/Bearer\s(\S+)/', $authHeader, $matches)) {
        respondWithError(401, "Yêu cầu xác thực token (Authorization header missing or invalid).");
    }

    $token = $matches[1];
    $decoded = decodeJwt($token);

    if ($decoded && isset($decoded['userID'])) {
        return $decoded;
    } else {
        respondWithError(401, "Token không hợp lệ hoặc đã hết hạn.");
    }
}
?>
