<?php
require_once 'config.php';        // Nạp file cấu hình
require_once 'db_connection.php'; // Nạp file kết nối DB và hàm tiện ích

// Hàm để lấy quyền (insert, view, edit, delete) của người dùng hoặc nhóm cho một bảng cụ thể
function getUserTablePermissions($userID, $groupID, $tableName) {
    $conn = getDbConnection(); // Lấy kết nối cơ sở dữ liệu
    $permissions = [
        'allowInsert' => PERM_NONE,
        'allowView' => PERM_NONE,
        'allowEdit' => PERM_NONE,
        'allowDelete' => PERM_NONE
    ];

    $userPermsFound = false;

    // 1. Kiểm tra quyền tùy chỉnh của người dùng (nếu có)
    $stmt = $conn->prepare("SELECT `allowInsert`, `allowView`, `allowEdit`, `allowDelete` FROM `" . MEMBERSHIP_USERPERMISSIONS_TABLE . "` WHERE `memberID` = ? AND `tableName` = ?");
    if ($stmt) {
        $stmt->bind_param("ss", $userID, $tableName);
        $stmt->execute();
        $result = $stmt->get_result();
        $userPerms = $result->fetch_assoc();
        $stmt->close();
        if ($userPerms) {
            $permissions = $userPerms; // Quyền tùy chỉnh của người dùng sẽ ghi đè quyền nhóm
            $userPermsFound = true;
        }
    } else {
        error_log("Error preparing user permissions query: " . $conn->error);
    }

    // 2. Nếu không có quyền tùy chỉnh cho người dùng, kiểm tra quyền của nhóm
    if (!$userPermsFound) {
        $stmt = $conn->prepare("SELECT `allowInsert`, `allowView`, `allowEdit`, `allowDelete` FROM `" . MEMBERSHIP_GROUPPERMISSIONS_TABLE . "` WHERE `groupID` = ? AND `tableName` = ?");
        if ($stmt) {
            $stmt->bind_param("is", $groupID, $tableName);
            $stmt->execute();
            $result = $stmt->get_result();
            $groupPerms = $result->fetch_assoc();
            $stmt->close();
            if ($groupPerms) {
                $permissions = $groupPerms;
            }
        } else {
            error_log("Error preparing group permissions query: " . $conn->error);
        }
    }

    $conn->close();
    return $permissions;
}

// Hàm kiểm tra quyền thực tế của người dùng cho một hành động (view, insert, edit, delete) trên một bản ghi cụ thể
function checkPermission($userInfo, $tableName, $action, $recordID = null, $ownerFieldName = null) {
    $conn = getDbConnection();
    $userID = $userInfo['userID'];
    $groupID = $userInfo['groupID'];

    $tablePermissions = getUserTablePermissions($userID, $groupID, $tableName);

    $allowed = false;
    $permLevel = PERM_NONE;

    // Xác định cấp độ quyền cho hành động cụ thể
    switch ($action) {
        case 'view':
            $permLevel = $tablePermissions['allowView'];
            break;
        case 'insert':
            $permLevel = $tablePermissions['allowInsert'];
            break;
        case 'edit':
            $permLevel = $tablePermissions['allowEdit'];
            break;
        case 'delete':
            $permLevel = $tablePermissions['allowDelete'];
            break;
    }

    // Logic kiểm tra quyền dựa trên cấp độ AppGini
    if ($permLevel == PERM_ALL) {
        $allowed = true; // Quyền tất cả
    } elseif ($permLevel == PERM_GROUP) {
        if ($action == 'insert') {
            $allowed = true; // Quyền thêm cho nhóm (người dùng trong nhóm có thể thêm)
        } elseif (($action == 'view' || $action == 'edit' || $action == 'delete') && $recordID && $ownerFieldName) {
            // Đối với view/edit/delete, kiểm tra bản ghi thuộc về nhóm của người dùng
            // Lấy tất cả memberID thuộc cùng groupID của người dùng hiện tại
            $stmt = $conn->prepare("SELECT COUNT(*) FROM `" . $tableName . "` WHERE `id` = ? AND `" . $ownerFieldName . "` IN (SELECT `memberID` FROM `" . MEMBERSHIP_USERS_TABLE . "` WHERE `groupID` = ?)");
            if ($stmt) {
                $stmt->bind_param("ii", $recordID, $groupID);
                $stmt->execute();
                $stmt->bind_result($count);
                $stmt->fetch();
                $stmt->close();
                if ($count > 0) {
                    $allowed = true;
                }
            } else {
                error_log("Error preparing group permission check query: " . $conn->error);
            }
        } else {
            // Nếu không có recordID hoặc ownerFieldName cho PERM_GROUP, giả định là được phép.
            // Việc lọc thực tế sẽ diễn ra ở getOwnershipWhereClause cho GET requests.
            $allowed = true;
        }
    } elseif ($permLevel == PERM_OWNER) {
        if ($action == 'insert') {
            $allowed = true; // Quyền thêm cho chủ sở hữu (người dùng có thể thêm)
        } elseif (($action == 'view' || $action == 'edit' || $action == 'delete') && $recordID && $ownerFieldName) {
            // Đối với view/edit/delete, kiểm tra bản ghi thuộc về người dùng hiện tại
            $stmt = $conn->prepare("SELECT COUNT(*) FROM `" . $tableName . "` WHERE `id` = ? AND `" . $ownerFieldName . "` = ?");
            if ($stmt) {
                $stmt->bind_param("is", $recordID, $userID);
                $stmt->execute();
                $stmt->bind_result($count);
                $stmt->fetch();
                $stmt->close();
                if ($count > 0) {
                    $allowed = true;
                }
            } else {
                error_log("Error preparing owner permission check query: " . $conn->error);
            }
        }
    }
    $conn->close();
    return $allowed;
}

// Hàm để tạo điều kiện WHERE cho truy vấn SELECT dựa trên quyền xem (allowView)
// Điều này giúp lọc các bản ghi mà người dùng có quyền xem (PERM_OWNER, PERM_GROUP)
function getOwnershipWhereClause($userInfo, $tableName, $ownerFieldName = null) {
    $userID = $userInfo['userID'];
    $groupID = $userInfo['groupID'];
    $tablePermissions = getUserTablePermissions($userID, $groupID, $tableName);

    $whereClause = "1=1"; // Mặc định là không có điều kiện lọc (tức là PERM_ALL)

    if ($tablePermissions['allowView'] == PERM_OWNER && $ownerFieldName) {
        // Chỉ xem các bản ghi mà người dùng hiện tại là chủ sở hữu
        $whereClause = "`" . $ownerFieldName . "` = '" . $userID . "'";
    } elseif ($tablePermissions['allowView'] == PERM_GROUP && $ownerFieldName) {
        // Chỉ xem các bản ghi mà chủ sở hữu thuộc cùng nhóm với người dùng hiện tại
        // Lấy tất cả memberID thuộc cùng groupID của người dùng hiện tại
        $whereClause = "`" . $ownerFieldName . "` IN (SELECT `memberID` FROM `" . MEMBERSHIP_USERS_TABLE . "` WHERE `groupID` = " . $groupID . ")";
    }
    // Nếu allowView là PERM_ALL, $whereClause vẫn là "1=1", tức là không thêm điều kiện lọc.

    return $whereClause;
}

// Hàm này để ánh xạ tên bảng với tên cột "owner" của nó trong cơ sở dữ liệu.
// LƯU Ý QUAN TRỌNG: Bạn CẦN TÙY CHỈNH HÀM NÀY dựa trên cách AppGini của bạn
// lưu trữ thông tin chủ sở hữu (ví dụ: cột `memberID` hoặc `created_by_memberID`).
// Nếu một bảng không có cột chủ sở hữu hoặc bạn không muốn áp dụng quyền sở hữu,
// hãy trả về `null`.
function getOwnerFieldName($tableName) {
    $ownerFieldsMap = [
        'your_table_name_1' => 'owner_memberID', // Ví dụ: bảng 'khach_hang' có cột 'owner_memberID'
        'your_table_name_2' => 'created_by',     // Ví dụ: bảng 'san_pham' có cột 'created_by'
        'invoices' => 'customer_id', // Nếu AppGini theo dõi chủ sở hữu qua khóa ngoại đến bảng khách hàng
        // Thêm các bảng và cột owner tương ứng của bạn vào đây
    ];

    // Trả về tên cột chủ sở hữu nếu tồn tại, ngược lại trả về null
    return isset($ownerFieldsMap[$tableName]) ? $ownerFieldsMap[$tableName] : null;
}

?>
