<?php
// Đặt header để cho phép truy cập từ mọi nguồn gốc (Cross-Origin Resource Sharing - CORS)
header("Access-Control-Allow-Origin: *");
// Định dạng phản hồi là JSON và mã hóa UTF-8
header("Content-Type: application/json; charset=UTF-8");
// Cho phép các phương thức HTTP (GET, POST, PUT, DELETE) và OPTIONS (cho pre-flight requests của CORS)
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
// Cho phép các headers tùy chỉnh
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With, X-API-KEY");

// Xử lý yêu cầu OPTIONS (pre-flight request của CORS)
// Trình duyệt gửi yêu cầu OPTIONS trước khi gửi các yêu cầu phức tạp (POST, PUT, DELETE)
// để kiểm tra xem server có cho phép cross-origin request hay không.
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Nạp các file cần thiết
require_once 'db_connection.php';  // Để kết nối DB và hàm respondWithError
require_once 'auth.php';           // Để xác thực người dùng và JWT
require_once 'permissions.php';    // Để kiểm tra quyền của AppGini
require_once 'config.php';         // Để lấy API_KEY và các hằng số khác

// --- Xác thực API Key ---
// Đây là lớp bảo mật đầu tiên, đảm bảo chỉ các ứng dụng được ủy quyền mới có thể truy cập API.
$providedApiKey = isset($_SERVER['HTTP_X_API_KEY']) ? $_SERVER['HTTP_X_API_KEY'] : '';
if ($providedApiKey !== API_KEY) {
    respondWithError(403, "API Key không hợp lệ."); // 403 Forbidden
}
// --- Kết thúc Xác thực API Key ---


// Lấy thông tin người dùng từ JWT token (sau khi API Key đã được xác thực)
// Hàm authenticateUser() sẽ tự động dừng script nếu token không hợp lệ hoặc thiếu.
$userInfo = authenticateUser();

// Lấy phương thức yêu cầu HTTP (GET, POST, PUT, DELETE)
$method = $_SERVER['REQUEST_METHOD'];

// Lấy tên bảng từ tham số 'table' trong URL query string
$tableName = isset($_GET['table']) ? $_GET['table'] : '';
if (empty($tableName)) {
    respondWithError(400, "Vui lòng cung cấp tên bảng thông qua tham số 'table'."); // 400 Bad Request
}

// Lấy ID bản ghi từ tham số 'id' trong URL query string (chỉ áp dụng cho GET/PUT/DELETE một bản ghi cụ thể)
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Lấy tên cột chủ sở hữu của bảng (ví dụ: 'owner_memberID') từ hàm cấu hình
$ownerFieldName = getOwnerFieldName($tableName);

// Lấy kết nối cơ sở dữ liệu
$conn = getDbConnection();

// Xử lý các yêu cầu dựa trên phương thức HTTP
switch ($method) {
    case 'GET':
        // --- Xử lý yêu cầu GET (Đọc dữ liệu) ---
        // Kiểm tra quyền 'view' của người dùng cho bảng này
        if (!checkPermission($userInfo, $tableName, 'view')) {
            respondWithError(403, "Bạn không có quyền xem dữ liệu trong bảng này.");
        }

        // Lấy điều kiện WHERE để lọc dữ liệu theo quyền sở hữu/nhóm
        $whereClause = getOwnershipWhereClause($userInfo, $tableName, $ownerFieldName);

        if ($id) {
            // Lấy một bản ghi cụ thể theo ID
            $stmt = $conn->prepare("SELECT * FROM `" . $tableName . "` WHERE `id` = ? AND " . $whereClause);
            if (!$stmt) {
                respondWithError(500, "Lỗi chuẩn bị truy vấn GET by ID: " . $conn->error);
            }
            $stmt->bind_param("i", $id); // Bind ID
            $stmt->execute();
            $result = $stmt->get_result();
            $item = $result->fetch_assoc(); // Lấy một bản ghi

            if ($item) {
                http_response_code(200);
                echo json_encode($item); // Trả về bản ghi dưới dạng JSON
            } else {
                respondWithError(404, "Không tìm thấy bản ghi với ID " . $id . " hoặc bạn không có quyền truy cập.");
            }
            $stmt->close();
        } else {
            // Lấy tất cả các bản ghi từ bảng (có áp dụng điều kiện lọc theo quyền sở hữu/nhóm)
            $query = "SELECT * FROM `" . $tableName . "` WHERE " . $whereClause;

            // TODO: Mở rộng cho join/datalist/child-parent:
            // Nếu bạn muốn hiển thị các trường liên quan từ các bảng khác (lookup fields, cha/con),
            // bạn sẽ cần sửa đổi truy vấn SELECT và thêm JOIN ở đây.
            // Ví dụ:
            // if ($tableName === 'orders') {
            //     $query = "SELECT o.*, c.customer_name FROM `orders` o JOIN `customers` c ON o.customer_id = c.id WHERE " . $whereClause;
            // }

            $result = $conn->query($query);
            if (!$result) {
                respondWithError(500, "Lỗi truy vấn GET all: " . $conn->error);
            }
            $items = [];
            while ($row = $result->fetch_assoc()) {
                $items[] = $row; // Thêm mỗi hàng vào mảng
            }
            http_response_code(200);
            echo json_encode($items); // Trả về mảng các bản ghi dưới dạng JSON
        }
        break;

    case 'POST':
        // --- Xử lý yêu cầu POST (Tạo bản ghi mới) ---
        // Kiểm tra quyền 'insert' của người dùng cho bảng này
        if (!checkPermission($userInfo, $tableName, 'insert')) {
            respondWithError(403, "Bạn không có quyền thêm dữ liệu vào bảng này.");
        }

        // Đọc dữ liệu JSON từ request body
        $data = json_decode(file_get_contents("php://input"), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            respondWithError(400, "Dữ liệu JSON không hợp lệ: " . json_last_error_msg());
        }
        if (empty($data)) {
            respondWithError(400, "Dữ liệu không được rỗng để tạo bản ghi.");
        }

        // Tự động gán cột chủ sở hữu (owner field) nếu bảng có quyền PERM_OWNER hoặc PERM_GROUP cho insert
        if ($ownerFieldName) {
            $tablePermissions = getUserTablePermissions($userInfo['userID'], $userInfo['groupID'], $tableName);
            if ($tablePermissions['allowInsert'] == PERM_OWNER || $tablePermissions['allowInsert'] == PERM_GROUP) {
                $data[$ownerFieldName] = $userInfo['userID']; // Gán ID người dùng hiện tại làm chủ sở hữu
            }
        }

        // Chuẩn bị truy vấn INSERT
        $columns = implode("`, `", array_keys($data)); // Lấy tên các cột, thêm backticks cho an toàn
        $placeholders = implode(", ", array_fill(0, count($data), "?")); // Tạo placeholders cho prepared statement
        $values = array_values($data); // Lấy giá trị của các cột
        $types = str_repeat("s", count($values)); // Giả định tất cả các giá trị là chuỗi ('s').
                                                 // Nếu bạn có các trường số ('i' cho integer, 'd' cho double)
                                                 // hoặc các trường ngày tháng, bạn cần điều chỉnh chuỗi này.

        $stmt = $conn->prepare("INSERT INTO `" . $tableName . "` (`" . $columns . "`) VALUES (" . $placeholders . ")");
        if (!$stmt) {
            respondWithError(500, "Lỗi chuẩn bị truy vấn POST: " . $conn->error);
        }
        $stmt->bind_param($types, ...$values); // Bind các giá trị vào prepared statement

        if ($stmt->execute()) {
            http_response_code(201); // 201 Created
            echo json_encode(["message" => "Bản ghi đã được tạo thành công.", "id" => $conn->insert_id]);
        } else {
            respondWithError(500, "Lỗi khi tạo bản ghi: " . $stmt->error);
        }
        $stmt->close();
        break;

    case 'PUT':
        // --- Xử lý yêu cầu PUT (Cập nhật bản ghi) ---
        if (!$id) {
            respondWithError(400, "Vui lòng cung cấp ID bản ghi để cập nhật.");
        }
        // Kiểm tra quyền 'edit' của người dùng cho bản ghi này
        if (!checkPermission($userInfo, $tableName, 'edit', $id, $ownerFieldName)) {
            respondWithError(403, "Bạn không có quyền sửa bản ghi này hoặc không tìm thấy bản ghi.");
        }

        // Đọc dữ liệu JSON từ request body
        $data = json_decode(file_get_contents("php://input"), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            respondWithError(400, "Dữ liệu JSON không hợp lệ: " . json_last_error_msg());
        }
        if (empty($data)) {
            respondWithError(400, "Dữ liệu không được rỗng để cập nhật bản ghi.");
        }

        // Chuẩn bị truy vấn UPDATE
        $setParts = [];
        $values = [];
        $types = "";

        foreach ($data as $key => $value) {
            // Không cho phép cập nhật cột 'id' hoặc cột chủ sở hữu ('ownerFieldName') qua API để tránh rủi ro bảo mật
            if ($key == 'id' || ($ownerFieldName && $key == $ownerFieldName)) continue;
            $setParts[] = "`" . $key . "` = ?";
            $values[] = $value;
            $types .= "s"; // Giả định tất cả các giá trị là chuỗi
        }
        $values[] = $id;    // Thêm ID của bản ghi vào cuối mảng giá trị
        $types .= "i";      // Kiểu dữ liệu cho ID là integer

        if (empty($setParts)) {
            respondWithError(400, "Không có dữ liệu hợp lệ để cập nhật.");
        }

        $stmt = $conn->prepare("UPDATE `" . $tableName . "` SET " . implode(", ", $setParts) . " WHERE `id` = ?");
        if (!$stmt) {
            respondWithError(500, "Lỗi chuẩn bị truy vấn PUT: " . $conn->error);
        }
        $stmt->bind_param($types, ...$values); // Bind các giá trị và ID

        if ($stmt->execute()) {
            if ($stmt->affected_rows > 0) {
                http_response_code(200); // 200 OK
                echo json_encode(["message" => "Bản ghi đã được cập nhật thành công."]);
            } else {
                // affected_rows = 0 có thể do bản ghi không tồn tại hoặc không có gì thay đổi
                respondWithError(404, "Không tìm thấy bản ghi với ID " . $id . " hoặc không có gì thay đổi.");
            }
        } else {
            respondWithError(500, "Lỗi khi cập nhật bản ghi: " . $stmt->error);
        }
        $stmt->close();
        break;

    case 'DELETE':
        // --- Xử lý yêu cầu DELETE (Xóa bản ghi) ---
        if (!$id) {
            respondWithError(400, "Vui lòng cung cấp ID bản ghi để xóa.");
        }
        // Kiểm tra quyền 'delete' của người dùng cho bản ghi này
        if (!checkPermission($userInfo, $tableName, 'delete', $id, $ownerFieldName)) {
            respondWithError(403, "Bạn không có quyền xóa bản ghi này hoặc không tìm thấy bản ghi.");
        }

        $stmt = $conn->prepare("DELETE FROM `" . $tableName . "` WHERE `id` = ?");
        if (!$stmt) {
            respondWithError(500, "Lỗi chuẩn bị truy vấn DELETE: " . $conn->error);
        }
        $stmt->bind_param("i", $id); // Bind ID

        if ($stmt->execute()) {
            if ($stmt->affected_rows > 0) {
                http_response_code(200); // 200 OK
                echo json_encode(["message" => "Bản ghi đã được xóa thành công."]);
            } else {
                respondWithError(404, "Không tìm thấy bản ghi với ID " . $id . " trong bảng " . $tableName);
            }
        } else {
            respondWithError(500, "Lỗi khi xóa bản ghi: " . $stmt->error);
        }
        $stmt->close();
        break;

    default:
        // Phương thức HTTP không được hỗ trợ
        respondWithError(405, "Phương thức " . $method . " không được hỗ trợ."); // 405 Method Not Allowed
        break;
}

$conn->close(); // Đóng kết nối cơ sở dữ liệu
?>
